# Copyright (C) Nov 2020 Mellanox Technologies Ltd. All rights reserved.
# Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenIB.org BSD license below:
#
#     Redistribution and use in source and binary forms, with or
#     without modification, are permitted provided that the following
#     conditions are met:
#
#      - Redistributions of source code must retain the above
#        copyright notice, this list of conditions and the following
#        disclaimer.
#
#      - Redistributions in binary form must reproduce the above
#        copyright notice, this list of conditions and the following
#        disclaimer in the documentation and/or other materials
#        provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# --

__all__ = ['FetchICMAddressError', 'CrawlerIteratorError',
           'CrawlerSessionError', 'RequiredArgumentError']


class FetchICMAddressError(Exception):
    """A class that represents fetching ICM address error.

    :param ICMAddress icm_addr: Invalid ICM address
    """
    address_format = '<segment={:#x}, vhca_id={:#x}, index={:#x}>'

    def __init__(self, icm_addr):
        self.icm_addr = icm_addr

    def __str__(self):
        address = self.address_format.format(*self.icm_addr)
        return 'failed to fetch steering data, ICM address: {}'.format(address)


class CrawlerIteratorError(Exception):
    """A class that represents crawler iterator error.

    :param str message: An error message
    """
    def __init__(self, message):
        self.message = message

    def __str__(self):
        return 'failed to crawl steering data, error: {}'.format(self.message)


class CrawlerSessionError(Exception):
    """A class that represents crawler session error.

    :param str message: An error message
    """
    def __init__(self, message):
        self.message = message

    def __str__(self):
        return 'failed to create steering crawler session, error: {}'.format(
            self.message)


class RequiredArgumentError(Exception):
    """A class that represents required argument error.

    :param str argument_name: The name of the required argument
    :param str prefix: A prefix string for the error message
    """
    error_format = '{} error: the following argument is required: {}'

    def __init__(self, argument_name, *, prefix=''):
        self.argument = argument_name
        self.prefix = prefix

    def __str__(self):
        return self.error_format.format(self.prefix, self.argument).strip()
