# Copyright (C) Nov 2020 Mellanox Technologies Ltd. All rights reserved.
# Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenIB.org BSD license below:
#
#     Redistribution and use in source and binary forms, with or
#     without modification, are permitted provided that the following
#     conditions are met:
#
#      - Redistributions of source code must retain the above
#        copyright notice, this list of conditions and the following
#        disclaimer.
#
#      - Redistributions in binary form must reproduce the above
#        copyright notice, this list of conditions and the following
#        disclaimer in the documentation and/or other materials
#        provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# --

from .loader import PacketLoader
from ..constants import PacketType
from ..packet import PCAPPacket

try:
    from mlxlibpcap import read_pcap
except ImportError:
    def read_pcap(path, filters):
        raise NotImplementedError('reading PCAP file requires an external '
                                  'package named \'mlxlibpcap\',\nthis package '
                                  'requires installing MFT with --with-pcap '
                                  'option.')

__all__ = ['PCAPLoader']


class PCAPLoader(PacketLoader):
    """A concrete class that represents packet loader for PCAP packets.

    :param str path: The path to file which contains packets
    :param str filters: A filtering expression for direct BPF usage
    """
    __product_type__ = PacketType.PCAP.value

    def __init__(self, path, *, filters=None):
        super().__init__(path)
        self.filters = filters

    def __iter__(self):
        for pkt_info in read_pcap(self.path, self.filters):
            yield PCAPPacket(*pkt_info)
